from dataclasses import dataclass
from flexoentity import FlexOID, FlexoEntity, EntityType, EntityState


@dataclass
class Domain(FlexoEntity):
    """
    I am a helper class to provide more information than just a
    domain abbreviation in FlexOID, doing mapping and management
    """
    ENTITY_TYPE = EntityType.DOMAIN
    subtype = "GENERAL"
    fullname: str = ""
    description: str = ""
    classification: str = "UNCLASSIFIED"

    @classmethod
    def default(cls):
        """I return an default instance of myself"""
        return cls("GENERIC")

    def __post_init__(self):
        super().__post_init__()

        # Generate FlexOID only if missing
        if not getattr(self, "flexo_id", None):
            self.flexo_id = FlexOID.safe_generate(
                domain="GENERAL",
                entity_type=EntityType.DOMAIN.value,   # 'D'
                estate=EntityState.DRAFT.value,      # 'D'
                text=self.text_seed,
                version=1,
            )

    @property
    def text_seed(self) -> str:
        """
        I provide a deterministic text seed for ID generation.
        FIXME: There might be a better text seed, json probably
        """
        return f"{self.fullname}|{self.classification}|{self.owner_id}"

    @property
    def domain_code(self) -> str:
        """
        I am self-domaining — I use my own subtype as domain code.
        """
        return self.subtype or "GENERAL"

    def to_dict(self):
        """I return a dictionary representing my state"""
        base = super().to_dict()
        base.update({
            "fullname": self.fullname,
            "description": self.description,
            "classification": self.classification,
        })
        return base

    @classmethod
    def from_dict(cls, data):
        """I return a new instance of myself, created from state provided by a dictionary"""
        return cls(
            fullname=data.get("fullname", ""),
            description=data.get("description", ""),
            classification=data.get("classification", "UNCLASSIFIED"),
        )
