"""
versioned_entity.py — Shared base class for all Flex-O entities.
Integrates with hardened id_factory.py and content fingerprint tracking.
"""
import json
import re
from enum import Enum, auto
from dataclasses import dataclass, field
from datetime import datetime, UTC
from typing import Optional
from abc import ABC, abstractmethod
import hashlib


from flexoentity.id_factory import FlexOID
from flexoentity import canonical_seed

# ──────────────────────────────────────────────────────────────────────────────
#  Entity Types
# ──────────────────────────────────────────────────────────────────────────────


class EntityType(Enum):
    QUESTION = auto()
    MEDIA = auto()
    CATALOG = auto()
    EXAM = auto()
    DATABASE = auto()
    CERTIFICATE = auto()
    DOMAIN = auto()
    
    def short(self) -> str:
        mapping = {
            EntityType.QUESTION: "Q",
            EntityType.MEDIA: "M",
            EntityType.DOMAIN: "DOM",
            EntityType.CATALOG:  "CAT",
            EntityType.EXAM:     "EX",
            EntityType.DATABASE: "DB",
            EntityType.CERTIFICATE: "CERT"
        }
        return mapping[self]

# ──────────────────────────────────────────────────────────────────────────────
#  States
# ──────────────────────────────────────────────────────────────────────────────


class EntityState(Enum):
    DRAFT = auto()
    APPROVED = auto()
    APPROVED_AND_SIGNED = auto()
    PUBLISHED = auto()
    OBSOLETE = auto()

    def short(self) -> str:
        """
        Return a one-letter abbreviation for the state, used in Flex-O IDs.
        """
        mapping = {
            EntityState.DRAFT: "D",
            EntityState.APPROVED: "A",
            EntityState.APPROVED_AND_SIGNED: "S",
            EntityState.PUBLISHED: "P",
            EntityState.OBSOLETE: "O",

        }
        return mapping[self]

    @classmethod
    def from_short(cls, char: str):
        """
        Inverse of .short(): restore the EntityState from its one-letter code.
        """
        reverse = {
            "D": cls.DRAFT,
            "A": cls.APPROVED,
            "S": cls.APPROVED_AND_SIGNED,
            "P": cls.PUBLISHED,
            "O": cls.OBSOLETE,
        }
        try:
            return reverse[char.upper()]
        except KeyError:
            raise ValueError(f"Unknown state abbreviation: {char}")

    def __str__(self):
        return self.name


@dataclass(kw_only=True)
class FlexoEntity(ABC):
    domain: str
    etype: EntityType 
    state: EntityState
    flexo_id: Optional[FlexOID] = field(default=None)
    signature: str = field(default_factory=str)
    origin: Optional[str] = field(default=None)
    
    OID_PATTERN = re.compile(
        r"^(?P<domain>[A-Z0-9]+)-(?P<etype>[A-Z]+)"
        r"(?P<date>\d{6,8})-(?P<hash>[0-9A-F]+)@(?P<version>\d{3})(?P<state>[A-Z])$"
    )

    def __str__(self) -> str:
        return f"{self.domain_code()}-{self.etype}{self.date}-{self.unique_hash}@{self.version:03d}{self.state}"

    @classmethod
    def from_string(cls, s: str) -> "FlexOID":
        """Rehydrate a FlexOID from its canonical string form."""
        m = cls.OID_PATTERN.match(s.strip())
        if not m:
            raise ValueError(f"Invalid FlexOID string: {s}")
        gd = m.groupdict()
        return cls(
            domain=gd["domain"],
            etype=gd["etype"],
            state=gd["state"],
        )
    @property
    @abstractmethod
    def text_seed(self) -> str:
        """Canonicalized text used for ID generation."""
        raise NotImplementedError("Subclasses must define text_seed property")

    def canonical_seed(self) -> str:
        return canonical_seed(self.text_seed)

    @classmethod
    @abstractmethod
    def default(cls):
        """Return a minimal valid instance of this entity (DRAFT state)."""
        raise NotImplementedError("Subclasses must implement default()")

    def domain_code(self) -> str:
        """Return canonical domain code for serialization and ID generation."""
        return self.domain.domain if hasattr(self.domain, "domain") else self.domain

    def __post_init__(self):
        """
        Generate ID and content fingerprint.
        
        All entities must carry a `.domain` attribute exposing a domain code string.
        This may be a `Domain` instance or a temporary wrapper used by the `Domain`
        class itself to avoid circular initialization.
        """
       
        self.flexo_id = FlexOID.generate(self.domain_code(),
                                         self.etype.short(),
                                         self.state.short(),
                                         self.text_seed,
                                         1)
        seed = canonical_seed(self.text_seed)
        self.signature = hashlib.blake2s(seed.encode("utf-8"), digest_size=8).hexdigest().upper()

    def __str__(self):
        return (
            f"{self.etype.name}({self.flexo_id}, {self.state.name}, "
            f"sig={self.signature}..., v{self.version})"
        )
    def to_dict(self):
        return {
            "domain": self.domain_code(),
            "etype": self.etype.name,
            "state": self.state.name,
            "flexo_id": str(self.flexo_id),
            "signature": self.signature,
            "origin": self.origin,
        }
    
    @classmethod
    def from_dict(cls, data):
        from flexoentity.domain import Domain  # avoid circular import
        domain_obj = Domain(
            domain=data["domain"],
            etype=EntityType.DOMAIN,
            state=EntityState.DRAFT,  # default when reconstructing context
        )
        obj = cls(
            domain=domain_obj,
            etype=EntityType[data["etype"]],
            state=EntityState[data["state"]],
        )
        obj.flexo_id = FlexOID.from_string(data["flexo_id"])
        obj.signature = data.get("signature", "")
        obj.origin = data.get("origin")
        return obj
      
    def to_json(self, *, indent: int | None = None) -> str:
        """Serialize entity (and its FlexOID) into JSON."""
        return json.dumps(self.to_dict(), indent=indent, ensure_ascii=False)

    @classmethod
    def from_json(cls, data_str: str) -> "FlexoEntity":
        """Deserialize from a JSON string."""
        data = json.loads(data_str)
        return cls.from_dict(data)

    @staticmethod
    def should_version(state) -> bool:
        """
        Determine if a given lifecycle state should trigger a version increment.

        Entities typically version when they move into more stable or
        externally visible states, such as APPROVED, SIGNED, or PUBLISHED.
        """

        return state in (
            EntityState.APPROVED,
            EntityState.APPROVED_AND_SIGNED,
            EntityState.PUBLISHED,
        )
    def _update_signature(self) -> str:
        """Always recompute the entity's content signature."""
        seed = self.canonical_seed()
        return hashlib.blake2s(seed.encode("utf-8"), digest_size=8).hexdigest().upper()

    def _update_fingerprint(self) -> bool:
        """Update FlexOID if the content signature changed."""
        new_sig = self._update_signature()
        if new_sig != self.signature:
            self.signature = new_sig
            self.flexo_id = FlexOID.generate(self.domain_code(),
                                             self.etype.short(),
                                             self.state.short(),
                                             self.text_seed,
                                             self.flexo_id.version)
            return True
        return False
   
    # ───────────────────────────────────────────────────────────────
    def _transition(self, target_state: EntityState):
        """Internal helper for state transitions with version and fingerprint checks."""
        if target_state == EntityState.OBSOLETE:
            self.state = target_state
            return

        # Check if version should bump
        if self.should_version(target_state):
            self._update_fingerprint()
            self.flexo_id = FlexOID.next_version(self.flexo_id)

        self.state = target_state
        self.updated_at = datetime.now(UTC)

    # ───────────────────────────────────────────────────────────────
    # Lifecycle transitions
    # ───────────────────────────────────────────────────────────────

    @property
    def version(self) -> int:
        """Extract numeric version from the FlexO-ID string."""
        try:
            return int(str(self.flexo_id).rsplit("@", 1)[-1])
        except (ValueError, AttributeError):
            return 1  # fallback if malformed or missing

    def bump_version(self):
        """Increment version number on the ID."""
        self.flexo_id = FlexOID.next_version(self.flexo_id)
        self.updated_at = datetime.now(UTC)

    def approve(self):
        """
        Move from DRAFT to APPROVED state.
        Draft entities receive a new permanent FlexOID with incremented version.
        """
        if self.state == EntityState.DRAFT:
            new_version = self.flexo_id.version + 1
            new_fid = FlexOID.generate(self.domain_code(),
                self.etype.short(),
                EntityState.APPROVED.short(),
                self.text_seed,
                version=new_version
            )
            self.previous_id = self.flexo_id  # optional: keep audit trail
            self.flexo_id = new_fid
            self.state = EntityState.APPROVED
            self.updated_at = datetime.now(UTC)
            return self
        raise ValueError("Only drafts can be approved")

    def sign(self):
        self._transition(EntityState.APPROVED_AND_SIGNED)

    def publish(self):
        if self.state == EntityState.APPROVED_AND_SIGNED:
            self._transition(EntityState.PUBLISHED)

    def obsolete(self):
        if self.state != EntityState.OBSOLETE:
            self._transition(EntityState.OBSOLETE)

    def clone_new_base(self):
        """Start new lineage when obsolete."""
        self.flexo_id = FlexOID.clone_new_base(
            self.domain_code(),
            self.etype.short(),
            self.state.short(),
            self.text_seed,
        )
        self.state = EntityState.DRAFT
        self.updated_at = datetime.now(UTC)

    # ───────────────────────────────────────────────────────────────
    # Integrity verification
    # ───────────────────────────────────────────────────────────────

    @staticmethod
    def verify_integrity(entity) -> bool:
        """Verify that an entity’s content signature matches its actual content."""
        expected_sig = hashlib.blake2s(
            canonical_seed(entity.text_seed).encode("utf-8"), digest_size=8
        ).hexdigest().upper()
        return expected_sig == entity.signature

    def allowed_transitions(self) -> list[str]:
        """Return a list of possible next state names (for GUI use)."""
        match self.state:
            case EntityState.DRAFT:
                return ["APPROVED"]
            case EntityState.APPROVED:
                return ["APPROVED_AND_SIGNED", "OBSOLETE"]
            case EntityState.APPROVED_AND_SIGNED:
                return ["PUBLISHED", "OBSOLETE"]
            case EntityState.PUBLISHED:
                return ["OBSOLETE"]
            case EntityState.OBSOLETE:
                return []
            case _:
                return []
